<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * This is the model class for table "meter_reading".
 *
 * @property int $id
 * @property int $user_id
 * @property int $service_category_id
 * @property string $previous_reading
 * @property string $current_reading
 * @property string $consumption
 * @property string $amount
 * @property string $status
 * @property string $reading_date
 * @property int $created_at
 * @property int $updated_at
 *
 * @property User $user
 * @property ServiceCategory $serviceCategory
 */
class MeterReading extends ActiveRecord
{
    const STATUS_NEW = 'Новая';
    const STATUS_ACCEPTED = 'Показания приняты';
    const STATUS_ERROR = 'Ошибка в переданных показаниях';
    const STATUS_PAID = 'Оплачено';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%meter_reading}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['user_id', 'service_category_id', 'current_reading', 'reading_date'], 'required'],
            [['user_id', 'service_category_id', 'created_at', 'updated_at'], 'integer'],
            [['previous_reading', 'current_reading', 'consumption', 'amount'], 'number'],
            [['status'], 'string'],
            [['reading_date'], 'safe'],
            [['current_reading'], 'validateCurrentReading'],
            [['user_id'], 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
            [['service_category_id'], 'exist', 'skipOnError' => true, 'targetClass' => ServiceCategory::class, 'targetAttribute' => ['service_category_id' => 'id']],
        ];
    }

    /**
     * Валидация текущих показаний
     */
    public function validateCurrentReading($attribute, $params)
    {
        if (!$this->hasErrors()) {
            if ($this->current_reading < $this->previous_reading) {
                $this->addError($attribute, 'Текущие показания не могут быть меньше предыдущих');
            }
            
            if ($this->current_reading == $this->previous_reading) {
                $this->addError($attribute, 'Текущие показания должны отличаться от предыдущих');
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'service_category_id' => 'Услуга',
            'previous_reading' => 'Предыдущие показания',
            'current_reading' => 'Текущие показания',
            'consumption' => 'Расход',
            'amount' => 'Сумма к оплате',
            'status' => 'Статус',
            'reading_date' => 'Дата передачи',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
{
    if (parent::beforeSave($insert)) {
        // Автоматически рассчитываем расход
        if ($this->current_reading && $this->previous_reading) {
            $this->consumption = round($this->current_reading - $this->previous_reading, 3);
        }
        
        // Автоматически рассчитываем сумму
        if ($this->consumption && $this->service_category_id) {
            $category = ServiceCategory::findOne($this->service_category_id);
            if ($category) {
                $this->amount = round($this->consumption * $category->tariff, 2);
            }
        }
        
        if ($insert) {
            $this->created_at = time();
            if (empty($this->status)) {
                $this->status = self::STATUS_NEW;
            }
            if (empty($this->reading_date)) {
                $this->reading_date = date('Y-m-d');
            }
        }
        $this->updated_at = time();
        
        return true;
    }
    return false;
}

    /**
     * Получает пользователя
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Получает категорию услуги
     */
    public function getServiceCategory()
    {
        return $this->hasOne(ServiceCategory::class, ['id' => 'service_category_id']);
    }

    /**
     * Получает все статусы
     */
    public static function getStatuses()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_ACCEPTED => 'Показания приняты',
            self::STATUS_ERROR => 'Ошибка в переданных показаниях',
            self::STATUS_PAID => 'Оплачено',
        ];
    }

    /**
     * Получает цвет для статуса
     */
    public function getStatusColor()
    {
        switch ($this->status) {
            case self::STATUS_NEW:
                return 'warning';
            case self::STATUS_ACCEPTED:
                return 'info';
            case self::STATUS_ERROR:
                return 'danger';
            case self::STATUS_PAID:
                return 'success';
            default:
                return 'secondary';
        }
    }

    /**
     * Получает последние показания пользователя для категории
     */
    public static function getLastReading($userId, $categoryId)
    {
        return self::find()
            ->where(['user_id' => $userId, 'service_category_id' => $categoryId])
            ->orderBy(['reading_date' => SORT_DESC, 'created_at' => SORT_DESC])
            ->one();
    }

    /**
     * Получает все показания пользователя
     */
    public static function getUserReadings($userId)
    {
        return self::find()
            ->where(['user_id' => $userId])
            ->orderBy(['reading_date' => SORT_DESC, 'created_at' => SORT_DESC])
            ->all();
    }

    /**
     * Получает сумму к оплате за месяц
     */
    public static function getMonthlyAmount($userId, $year, $month)
    {
        $startDate = date('Y-m-01', strtotime("$year-$month-01"));
        $endDate = date('Y-m-t', strtotime("$year-$month-01"));
        
        $readings = self::find()
            ->where(['user_id' => $userId])
            ->andWhere(['between', 'reading_date', $startDate, $endDate])
            ->all();
        
        $total = 0;
        foreach ($readings as $reading) {
            $total += $reading->amount;
        }
        
        return $total;
    }
}