<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\MeterReading;
use app\models\User;
use app\models\News;
use app\models\ServiceCategory;

class AdminController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'matchCallback' => function ($rule, $action) {
                            return !Yii::$app->user->isGuest && Yii::$app->user->identity->phone === '8(000)000-00-00';
                        },
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    Yii::$app->session->setFlash('error', 'Доступ запрещен. Только для администраторов.');
                    return Yii::$app->getResponse()->redirect(['site/login']);
                },
            ],
        ];
    }

    /**
     * Панель администратора - список всех заявок
     */
    public function actionIndex()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => MeterReading::find()
                ->joinWith(['user', 'serviceCategory'])
                ->orderBy(['created_at' => SORT_DESC]),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Просмотр и изменение статуса заявки
     */
    public function actionView($id)
    {
        $model = MeterReading::findOne($id);
        
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Заявка не найдена.');
        }

        if ($model->load(Yii::$app->request->post())) {
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Статус заявки обновлен.');
                return $this->redirect(['index']);
            }
        }

        return $this->render('view', [
            'model' => $model,
        ]);
    }

    /**
     * Управление пользователями
     */
    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find(),
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Управление новостями
     */
    public function actionNews()
    {
        $model = new News();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Новость добавлена.');
            return $this->refresh();
        }

        $news = News::getAllNews();

        return $this->render('news', [
            'model' => $model,
            'news' => $news,
        ]);
    }

    /**
     * Удаление новости
     */
    public function actionDeleteNews($id)
    {
        $model = News::findOne($id);
        
        if ($model && $model->delete()) {
            Yii::$app->session->setFlash('success', 'Новость удалена.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка при удалении новости.');
        }

        return $this->redirect(['news']);
    }

    /**
     * Управление категориями услуг
     */
    public function actionServices()
    {
        $model = new ServiceCategory();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Категория добавлена.');
            return $this->refresh();
        }

        $categories = ServiceCategory::getAllCategories();

        return $this->render('services', [
            'model' => $model,
            'categories' => $categories,
        ]);
    }

    /**
     * Статистика
     */
    public function actionStats()
    {
        $totalUsers = User::find()->count();
        $totalReadings = MeterReading::find()->count();
        $totalAmount = MeterReading::find()->sum('amount');
        $pendingReadings = MeterReading::find()->where(['status' => MeterReading::STATUS_NEW])->count();

        // Статистика по месяцам
        $monthlyStats = MeterReading::find()
            ->select([
                'DATE_FORMAT(reading_date, "%Y-%m") as month',
                'COUNT(*) as count',
                'SUM(amount) as total'
            ])
            ->groupBy('month')
            ->orderBy('month DESC')
            ->limit(6)
            ->asArray()
            ->all();

        return $this->render('stats', [
            'totalUsers' => $totalUsers,
            'totalReadings' => $totalReadings,
            'totalAmount' => $totalAmount,
            'pendingReadings' => $pendingReadings,
            'monthlyStats' => $monthlyStats,
        ]);
    }
}